/* 

	Suitable for PIC18 devices, 
	
	modified & optimised by Mauro Grassi,
	
	from original by Mauro Grassi, 2010.

*/

#include "sd.h"
#include "crc.h"
#include "spi.h"
#include "HardwareProfile.h"
#include "Delay.h"
#include "ff.h"
#include "mystring.h"
#include <math.h>
#include "main.h"
#include "serialUSB.h"
#include "diskio.h"
#include "uart.h"

extern FATFS 	fileSystem;

#pragma udata sddata

#pragma code usercode

#if 1
int sendCommandSDCardSPI(BYTE cmd, BYTE response_type, BYTE* response, BYTE *argument)
{
	TIMEOUT_TYPE 	i;
	unsigned char 	tmp;
	unsigned char   response_length;
	BYTE 			ibuffer[6];
		
	ibuffer[0]=((cmd & 0x3F) | 0x40);
	ibuffer[1]=argument[0];
	ibuffer[2]=argument[1];
	ibuffer[3]=argument[2];
	ibuffer[4]=argument[3];
	
	#if(USE_CRC_ON_COMMANDS)
	
	ibuffer[5]=CRC7(&ibuffer[0], 5);
	
	#else
	
	/* Must Always Use CRC for CMD8 and 0x95 is the default for CMD0 */
	
	if((cmd & 0x3F)==CMD8)
	{
		ibuffer[5]=CRC7(&ibuffer[0], 5);
	}
	else
	{
		ibuffer[5]=0x95;
	}
	#endif

	switch (response_type)
	{

		case R1:
		case R1B:
			response_length = 1;
			break;

		case R2:
			response_length = 2;
			break;

		case R3:
			response_length = 5;
			break;

		case R7:
			response_length = 5;
			break;

		case R1X18:
			response_length = 19;
			break;

		case R2X64:
			response_length = 68;
			break;
		
		default:
			response_length = 0;
			break;
	}
	
	#if (DEBUG_SD_INIT)
		putrsUART((const rom char*)"Send CMD: ");
		for(i=0; i<6; i++)
		{
			disAUART(ibuffer[i]);
			putrsUART((const   rom char*)".");		
		}
		putrsUART((const   rom char*)"\r\n");
	#endif
		
	// SDCS=0;
	/* All data is sent MSB first, and MSb first */
	/* Send the header/command */
	/* Format:
	
	cmd[7:6] : 01
	cmd[5:0] : command 
	
	*/
	#if(USE_PIC18_DMA)

		DMACON1bits.TXINC=1;	/* increment Tx pointer */
		DMACON1bits.RXINC=0;	/* do not increment Rx pointer */
		DMABCH=0x00;
		DMABCL=0x05;			/* choose 6 bytes */
		TXADDRH=((unsigned int)&ibuffer[0])>>8;
		TXADDRL=((unsigned int)&ibuffer[0]);
		RXADDRH=((unsigned int)&SPI18SmallBuffer[0])>>8;
		RXADDRL=((unsigned int)&SPI18SmallBuffer[0]);
		SDCS=0;
		DMACON1bits.DMAEN=1;
		while(DMACON1bits.DMAEN);
		SDCS=1;

	#else
		
		for (i=0; i<=5; i++)
		{
			WriteSPI(ibuffer[i]);
		}
	
	#endif

	/* 
		Wait for a response. A response can be recognized by the
		start bit (a zero) 
	
	*/	
	i=0;
	SDCS=0;
	do
	{
		tmp=WriteSPIWithoutSS(0xFF);
		i++;
	}
	while((tmp & 0x80)&&(i<SDMMC_CMD_TIMEOUT));
	SDCS=1;
	
	/* Just bail if we never got a response */
	if(i>=SDMMC_CMD_TIMEOUT)return 0;

	if(response_type==R1X18)
	{
		// this type of response is for SPI Mode only and only for CMD9 and CMD10 that
		// send the CID and CSD registers back in a 16 byte + 2 byte packet
		// the 16 byte packet contains the register while the 2 byte suffix is the CRC16 field.
		// the first response is a R1 type, then we wait for the card to response by polling
		// bit 0.
		response[0]=tmp;
		i=0;
		SDCS=0;
		do
		{
			tmp=WriteSPIWithoutSS(0xFF);
			i++;
		}
		while ((tmp & 1) && (i < SDMMC_CMD_TIMEOUT));
		tmp=WriteSPIWithoutSS(0xFF);
   		SDCS=1;
   		i=1;
    }
	else
	if(response_type==R2X64)
	{
		// for the ACMD13 response which is R2+64 bytes long!
		response[0]=tmp;
		SDCS=0;
		response[1]=WriteSPIWithoutSS(0xFF);
		i=0;
		do
		{
		tmp=WriteSPIWithoutSS(0xFF);
		i++;
		}
		while ((tmp & 1) && (i < SDMMC_CMD_TIMEOUT));
		tmp=WriteSPIWithoutSS(0xFF);
   		SDCS=1;
   		i=2;
   	}
   	else
   	{
   		i=0;
   	} 
	
	if(response_length>1)
	{
		SDCS=0;
		while(i<response_length)
		{
		 response[i++] = tmp;
		 /* This handles the trailing-byte requirement. */
		 tmp = WriteSPIWithoutSS(0xFF);
		}  
		SDCS=1;
	}
	else
	{
		response[i++]=tmp;
		tmp=WriteSPI(0xFF);
	}	

	/* If the response is a "busy" type (R1B), then there's some
	* special handling that needs to be done. The card will
	* output a continuous stream of zeros, so the end of the BUSY
	* state is signaled by any nonzero response. The bus idles
	* high.
	*/
	
	if (response_type == R1B)
	{
		SDCS=0;
		do	
		{
			tmp=WriteSPIWithoutSS(0xFF);
		}
		/* This should never time out, unless SDI is grounded.
		   Don't bother forcing a timeout condition here. */
		while (tmp != 0xFF);
		WriteSPIWithoutSS(0xFF);
		SDCS=1;
	}
	
	#if (DEBUG_SD_INIT)
		putrsUART((const   rom char*)"Receive : ");
		for(i=0; i<response_length; i++)
		{
			disAUART(response[i]);
			putrsUART((const   rom char*)".");
		}
		putrsUART((const   rom char*)"\r\n");
	#endif
	return 1;
}
#endif

#if(USE_SINGLE_BLOCK_RW)
unsigned int readSingleBlockSDCardSPI(CARD_INFO* thisCard, unsigned long blockaddr, unsigned int response_length, BYTE* data, BYTE* response)
{
	TIMEOUT_TYPE  	i;
	unsigned char  	tmp;
	BYTE 			ibuffer[6];
	unsigned int 	CRCResult;
	unsigned int 	CRCReceived;
	BYTE* 			dataIn;
	unsigned long 	bytesRead;
	int retries = 3;
	
	dataIn=data;
	do {
		ibuffer[1]=(blockaddr>>24);
		ibuffer[2]=(blockaddr>>16);
		ibuffer[3]=(blockaddr>>8);
		ibuffer[4]=blockaddr;
		sendCommandSDCardSPI(CMD17, CMD17_R, &ibuffer[0], &ibuffer[1]);
	} while( ibuffer[0] != 0 && retries-- );
	*response++=ibuffer[0];
	
	i=0;
	SDCS=0;
	do
	{
		tmp=WriteSPIWithoutSS(0xFF);
		i++;
	}
	// 0xFE= start token
	while((tmp!=0xFE)&&(i<SDMMC_READ_TIMEOUT));
	SDCS=1;
		
	#if (DEBUG_SD_INIT)
	if(i>=SDMMC_READ_TIMEOUT)
	{
		putrsUART((const   rom char*)"TIMEOUT\r\n");
	}
	#endif
	
	if(i>=SDMMC_READ_TIMEOUT)
	{
		return 0;
	}

#if (USE_SD_DMA)

	i=response_length;
	if(i == 512 ) {
		BulkReadSPI();
		inptr=(unsigned char*)&SPIBigBuffer[0];
		while(i--)
		{
			*data++=*inptr++;
		}
	} 
	else 
	{
		i=response_length;
		while(i--)
		{
		*data++=WriteSPI(0xFF);
		}
	}

#elif(USE_PIC18_DMA)

	if(response_length<=512)
	{
		(void)BulkReadPIC18SPI(data, response_length);
	}
	else
	{
		i=response_length;
		SDCS=0;
		while(i--)
		{
			*data++=WriteSPIWithoutSS(0xFF);
		}	
		SDCS=1;
	}

#else
	
	i=response_length;
	SDCS=0;
	while(i--)
	{
	*data++=WriteSPI(0xFF);
	}	
	SDCS=1;
			
#endif

	#if (DEBUG_SD_INIT)
		putrsUART((const   rom char*)"Read    : ");
		for(i=0; i<512; i++)
		{
			disAUART(*dataIn);
			dataIn++;
			putrsUART((const   rom char*)".");
			if((i & 0x0F)==0x0F)putrsUART((const   rom char*)"\r\nRead    : ");
		}
		putrsUART((const   rom char*)"\r\n");
	#endif

	SDCS=0;
	tmp=WriteSPIWithoutSS(0xFF);
	CRCReceived=WriteSPIWithoutSS(0xFF); // CRC16
	CRCReceived+=(tmp<<8);
	SDCS=1;
	
	#if(USE_CRC_ON_DATA)
	
	if(thisCard->checkCRC)CRCResult=CRC16(dataIn, response_length); else CRCResult=CRCReceived;
	
	*response++=(CRCResult>>8);
	*response++=(CRCResult);
	*response++=(CRCReceived>>8);
	*response++=(CRCReceived);
	
	#else
	
	CRCResult=CRCReceived;
	
	#endif
	
	// the following "fixes" the 2Gb card problems since all 2Gb SD cards report 1024 block size instead of 512.
	// also used for SDHC cards...
	
	if((thisCard->csd.READ_BL_LEN!=0x09)||((thisCard->CARD_TYPE & 6)==6))
	{
		ibuffer[1]=0;
		ibuffer[2]=0;
		ibuffer[3]=0;
		ibuffer[4]=0;
		sendCommandSDCardSPI(CMD12, CMD12_R, &ibuffer[0], &ibuffer[1]);
	}
	return response_length;
}
#endif

unsigned long readMultipleBlockSDCardSPI(CARD_INFO* thisCard, unsigned long blockaddr, unsigned int response_length, BYTE* data, BYTE* response, unsigned int numBlocks)
{
	unsigned char 	tmp;
	TIMEOUT_TYPE 	i;
	BYTE 			ibuffer[6];

	#if(USE_CRC_ON_DATA)
	unsigned int 	CRCResult;
	unsigned int 	CRCGood;
	#endif
	unsigned int 	CRCReceived;
	
	BYTE* 			dataIn;
	unsigned long 	bytesRead;
	int retries = 3;

	do {
		ibuffer[1]=(blockaddr>>24);
		ibuffer[2]=(blockaddr>>16);
		ibuffer[3]=(blockaddr>>8);
		ibuffer[4]=blockaddr;
		sendCommandSDCardSPI(CMD18, CMD18_R, &ibuffer[0], &ibuffer[1]);
	} while( ibuffer[0] != 0 && retries-- );
	*response++=ibuffer[0];

	bytesRead=0;
	#if(USE_CRC_ON_DATA)
	CRCGood=0;
	#endif
	dataIn=data;
	while(numBlocks>0)
	{
		i=0;
		SDCS=0;
		do
		{
			tmp=WriteSPIWithoutSS(0xFF);
			i++;
		}
		// 0xFE= start token
		while ((tmp!=0xFE) && (i<SDMMC_READ_TIMEOUT));
		SDCS=1;
		
		if(i>=SDMMC_READ_TIMEOUT)
		{
			return 0;
		}
			
		#if (USE_SD_DMA)
	
			i=response_length;
			if(i == 512 ) {
				BulkReadSPI();
				inptr=(unsigned char*)&SPIBigBuffer[0];
				while(i--)
				{
					*data++=*inptr++;
				}
			} else 
			{
				i=response_length;
				while(i--)
				{
				*data++=WriteSPI(0xFF);
				}
			}

		#elif(USE_PIC18_DMA)
		
			if(response_length<=512)
			{
				(void)BulkReadPIC18SPI(data, response_length);
				data+=response_length;
			}
			else
			{
				i=response_length;
				SDCS=0;
				while(i--)
				{
					*data++=WriteSPIWithoutSS(0xFF);
				}
				SDCS=1;
			}
		
		#else
				i=response_length;
				SDCS=0;
				while(i--)
				{
					*data++=WriteSPIWithoutSS(0xFF);
				}
				SDCS=1;		
		#endif
		
		SDCS=0;		
		CRCReceived=((unsigned short)WriteSPIWithoutSS(0xFF)<<8);
		CRCReceived+=WriteSPIWithoutSS(0xFF);			// CRC16
		SDCS=1;
		
	#if(USE_CRC_ON_DATA)
	
		if(thisCard->checkCRC)CRCResult=CRC16(dataIn, response_length); else CRCResult=CRCReceived;
		if(CRCResult==CRCReceived)CRCGood++;
		
	#else
	
	#endif

		dataIn+=response_length;
		numBlocks--;
		bytesRead+=response_length;
	}
	
	ibuffer[1]=0;
	ibuffer[2]=0;
	ibuffer[3]=0;
	ibuffer[4]=0;
	sendCommandSDCardSPI(CMD12, CMD12_R, &ibuffer[0], &ibuffer[1]);
	#if(USE_CRC_ON_DATA)
	*response++=(unsigned char)(CRCGood>>8);
	*response++=(unsigned char)(CRCGood);
	#endif
	return (bytesRead);
}

#if(USE_SINGLE_BLOCK_RW)
unsigned int writeSingleBlockSDCardSPI(CARD_INFO* thisCard, unsigned long blockaddr, unsigned int response_length, BYTE* data, BYTE* response)
{
	unsigned char 	tmp;
	TIMEOUT_TYPE 	i;
	BYTE 			ibuffer[6];
	
	unsigned int 	CRCResult;
	unsigned int 	CRCReceived;
	
	#if(USE_CRC_ON_DATA)
	
	#endif
	
	BYTE* 			dataIn;
	unsigned long 	bytesWritten;
	int retries = 3;
   	
	// returns the number of bytes written
	#if(USE_CRC_ON_DATA)
	
		if(thisCard->checkCRC)CRCResult=CRC16(data, response_length); else CRCResult=0;
	
	#else
	
		CRCResult=0;
	
	#endif

	do {
		ibuffer[1]=(blockaddr>>24);
		ibuffer[2]=(blockaddr>>16);
		ibuffer[3]=(blockaddr>>8);
		ibuffer[4]=blockaddr;
		sendCommandSDCardSPI(CMD24, CMD24_R, &ibuffer[0], &ibuffer[1]);
	} while( ibuffer[0] != 0 && retries-- );
	*response++=ibuffer[0];
	
	if((ibuffer[0] & 1)==0)
	{
		SDCS=0;
		WriteSPIWithoutSS(0xFF);					// one byte gap!
		WriteSPIWithoutSS(0xFE);					// start token!
		SDCS=1;
		
	#if (USE_SD_DMA)
	
		i=response_length;
		if(i == 512 ) 
			{
			inptr=(unsigned char*)&SPIBigBuffer[0];
			while(i--)
			{
				*inptr++=*data++;
			}
			BulkWriteSPI();
		} else {
	
		i=response_length;
		while(i--)
		{
			tmp=WriteSPI(*data);
			data++;
		}	
	
		}
	#elif(USE_PIC18_DMA)

		if(response_length<=512) 
		{
				
			(void)BulkWritePIC18SPI(data, response_length);
						
		} 
		else 
		{
	
			i=response_length;
			SDCS=0;
			while(i--)
			{
				tmp=WriteSPIWithoutSS(*data++);
			}
			SDCS=1;
	
		}
	#else
	
			i=response_length;
			SDCS=0;
			while(i--)
			{
				tmp=WriteSPIWithoutSS(*data++);
			}
			SDCS=1;

	#endif
	
		SDCS=0;
		WriteSPIWithoutSS(CRCResult>>8);
		WriteSPIWithoutSS(CRCResult);	// CRC16
		SDCS=1;
		tmp=WriteSPI(0xFF);				// get data response token into tmp			

		// the data response token has the following format:
		// bit 	7	6	5	4	3	2	1	0
		// 		X	X	X 	0	S2	S1	S0	1	where S2, S1, S0 = 	010 = Data Accepted
		//															101	= Data Rejected due to CRC error
		//															110 = Data Rejected due to a Write Error
		
		*response++=tmp;
		*response++=(CRCResult>>8);
		*response++=(CRCResult);	

		i=0;
		SDCS=0;
		tmp=WriteSPIWithoutSS(0xFF);
		do
		{
			tmp=WriteSPIWithoutSS(0xFF);				// initiate the writing... wait till the card is finished, ie not busy!, busy is a trailing 0 stream whereas not busy is a trailing 1 stream of bits
			i++;
		}
		while((tmp==0)&&(i<SDMMC_WRITE_TIMEOUT));
		WriteSPIWithoutSS(0xFF);
		SDCS=1;

	    if(i>=SDMMC_WRITE_TIMEOUT)
	    {
			return 0;
		}	
		return response_length;
	}
	return 0;
}
#endif

unsigned long writeMultipleBlockSDCardSPI(CARD_INFO* thisCard, unsigned long blockaddr, unsigned int response_length, BYTE* data, BYTE* response, unsigned int numBlocks)
{
	unsigned char 	tmp;
	TIMEOUT_TYPE 	i;
	BYTE 			ibuffer[6];
	unsigned int 	CRCResult;
	unsigned int 	CRCReceived;
	unsigned int    CRCGood;
	BYTE* 			dataIn;
	unsigned long 	bytesWritten;
	int retries = 3;

	// returns the number of bytes written

	do {
		ibuffer[1]=(blockaddr>>24);
		ibuffer[2]=(blockaddr>>16);
		ibuffer[3]=(blockaddr>>8);
		ibuffer[4]=blockaddr;
		sendCommandSDCardSPI(CMD25, CMD25_R, &ibuffer[0], &ibuffer[1]);
	} while( ibuffer[0] != 0 && retries-- );
	*response++=ibuffer[0];
	
	if((ibuffer[0] & 1)==0)
	{
		WriteSPI(0xFF);			// one byte gap!
		bytesWritten=0;
		CRCGood=0;
		while(numBlocks)
		{
		
		SDCS=0;
		WriteSPIWithoutSS(0xFF);
		WriteSPIWithoutSS(0xFC);			// START MULTIPLE BLOCK WRITE token!
		SDCS=1;	
		
		#if(USE_CRC_ON_DATA)
		
		if(thisCard->checkCRC)CRCResult=CRC16(data, response_length); else CRCResult=0;
		
		#else
		
		CRCResult=0;
		
		#endif

	#if (USE_SD_DMA)
	
		i=response_length;
		if(i == 512 ) 
			{
			inptr=(unsigned char*)&SPIBigBuffer[0];
			while(i--)
			{
				*inptr++=*data++;
			}
			BulkWriteSPI();
		} else {
	
		i=response_length;
		while(i--)
		{
			tmp=WriteSPI(*data);
			data++;
		}	
	
		}
		
	#elif(USE_PIC18_DMA)

		if(response_length<=512) 
			{
				(void)BulkWritePIC18SPI(data, response_length);
				data+=response_length;	
			} 
			else 
			{
				i=response_length;
				SDCS=0;
				while(i--)
				{
					tmp=WriteSPIWithoutSS(*data++);
				}	
				SDCS=1;
			}

	#else
	
   				i=response_length;
				SDCS=0;
				while(i--)
				{
					tmp=WriteSPIWithoutSS(*data++);
				}	
				SDCS=1;

   	#endif
   	
   		SDCS=0;
		WriteSPIWithoutSS(CRCResult>>8);
		WriteSPIWithoutSS(CRCResult);		// CRC16
		SDCS=1;
		tmp=WriteSPI(0xFF);		// get data response token into tmp
	
		// the data response token has the following format:
		// bit 	7	6	5	4	3	2	1	0
		// 		X	X	X 	0	S2	S1	S0	1	where S2, S1, S0 = 	010 = Data Accepted
		//															101	= Data Rejected due to CRC error
		//															110 = Data Rejected due to a Write Error
		if((tmp & 0x0F)==0x0B)
		{
		
		}
		else
		{
			CRCGood++;
		}
		
		i=0;
		SDCS=0;
		tmp=WriteSPIWithoutSS(0xFF);
		do
		{
		tmp=WriteSPIWithoutSS(0xFF);				// initiate the writing... wait till the card is finished, ie not busy!
													// busy is a trailing 0 stream whereas not busy is a trailing 1 stream of bits
		i++;
		}
		while ((tmp==0) && (i<SDMMC_WRITE_TIMEOUT));
      	SDCS=1;
      	
   		if(i>=SDMMC_WRITE_TIMEOUT)
   		{
    		return 0;
		}
		bytesWritten+=response_length;
		numBlocks--;
		}
		
		SDCS=0;
		WriteSPIWithoutSS(0xFD);					// STOP TRANSFER TOKEN
		WriteSPIWithoutSS(0xFF);
		SDCS=1;
		
		tmp=0;
		i=0;
		SDCS=0;
		do
		{
		tmp=WriteSPIWithoutSS(0xFF);				// initiate the writing... wait till the card is finished, ie not busy!
													// busy is a trailing 0 stream whereas not busy is a trailing 1 stream of bits
		i++;
		}
		while ((tmp==0) && (i<SDMMC_WRITE_TIMEOUT));
		WriteSPIWithoutSS(0xFF);	
		SDCS=1;
		
		*response++=(unsigned char)(CRCGood>>8);
		*response++=(unsigned char)(CRCGood);
		if(i>=SDMMC_WRITE_TIMEOUT)
   		{
			return 0;
		}
		return bytesWritten;
	}
	return 0;
}

#if(USE_SD_REGISTERS!=0)
int getCIDRegister(CARD_INFO* thisCard)
{
	int i, j;
	unsigned char	receiveBuffer[20];

	for(i=0; i<20; i++)receiveBuffer[i]=0;
	sendCommandSDCardSPI(CMD10, CMD10_R, &receiveBuffer[0], &receiveBuffer[0]);
	// now receiveBuffer[1]-receiveBuffer[16] contains the CID register 128 bits...
	if(thisCard->CARD_TYPE & 6)
	{
		// SDHC or SD Cards
		thisCard->cid.MID=receiveBuffer[1];
		thisCard->cid.OID=(receiveBuffer[2]<<8) + receiveBuffer[3];
		thisCard->cid.PNM[0]=receiveBuffer[4];
		thisCard->cid.PNM[1]=receiveBuffer[5];
		thisCard->cid.PNM[2]=receiveBuffer[6];
		thisCard->cid.PNM[3]=receiveBuffer[7];
		thisCard->cid.PNM[4]=receiveBuffer[8];
		thisCard->cid.PNM[5]='\0';
		thisCard->cid.PRV=receiveBuffer[9];
		thisCard->cid.PSN[0]=receiveBuffer[10];
		thisCard->cid.PSN[1]=receiveBuffer[11];
		thisCard->cid.PSN[2]=receiveBuffer[12];
		thisCard->cid.PSN[3]=receiveBuffer[13];
		thisCard->cid.MDT=(receiveBuffer[14]<<8)+receiveBuffer[15];
	} else
	if(thisCard->CARD_TYPE==1)
	{
		// MMC
		thisCard->cid.MID=receiveBuffer[1];
		thisCard->cid.OID=(receiveBuffer[2]<<8) + receiveBuffer[3];
		thisCard->cid.PNM[0]=receiveBuffer[4];
		thisCard->cid.PNM[1]=receiveBuffer[5];
		thisCard->cid.PNM[2]=receiveBuffer[6];
		thisCard->cid.PNM[3]=receiveBuffer[7];
		thisCard->cid.PNM[4]=receiveBuffer[8];
		thisCard->cid.PNM[5]=receiveBuffer[9];
		thisCard->cid.PNM[6]='\0';

		thisCard->cid.PRV=receiveBuffer[10];
		thisCard->cid.PSN[0]=receiveBuffer[11];
		thisCard->cid.PSN[1]=receiveBuffer[12];
		thisCard->cid.PSN[2]=receiveBuffer[13];
		thisCard->cid.PSN[3]=receiveBuffer[14];
		// now MMC dates are in the form MM (high nibble: month) YY (low nibble: year offset from 1997)
		// so we convert to a SD card date that is YYM (YY: year offset from 2000, M: month nibble)
		i=receiveBuffer[15]>>4;			// i contains the month
		j=(receiveBuffer[15] & 0x0F)-3; // -3 to offset to 2000 instead...
		thisCard->cid.MDT=i + (j<<4);
	} else
	{
		// unknown!
		thisCard->cid.MID=0;
		thisCard->cid.OID=0;
		thisCard->cid.PNM[0]=0;
		thisCard->cid.PNM[1]=0;
		thisCard->cid.PNM[2]=0;
		thisCard->cid.PNM[3]=0;
		thisCard->cid.PNM[4]=0;
		thisCard->cid.PNM[5]='\0';
		thisCard->cid.PRV=0;
		thisCard->cid.PSN[0]=0;
		thisCard->cid.PSN[1]=0;
		thisCard->cid.PSN[2]=0;
		thisCard->cid.PSN[3]=0;
		thisCard->cid.MDT=0;
	}
	return 1;
}

int getCSDRegister(CARD_INFO* thisCard)
{
	// Now we read the CSD register
	int i, j;
	unsigned char	receiveBuffer[20];
	
	for(i=0; i<20; i++)receiveBuffer[i]=0;
	sendCommandSDCardSPI(CMD9, CMD9_R, &receiveBuffer[0], &receiveBuffer[0]);
	// now receiveBuffer[1]-receiveBuffer[16] contains the CSD register 128 bits
	if((receiveBuffer[1] & 0xC0)==0)
	{
		// for Ver 1.0 SD or Ver 2. SD Card (<=2Gb) ie, not SDHC card
		// this shows a 00b in the first two bits (bit 7 and bit 6 of receiveBuffer[1]) of the CSD register
		thisCard->csd.TAAC=receiveBuffer[2];
		thisCard->csd.NSAC=receiveBuffer[3];
		thisCard->csd.TRAN_SPEED=receiveBuffer[4];
		thisCard->csd.CCC=((unsigned short)receiveBuffer[5]<<4)+((unsigned short)receiveBuffer[6]>>4);
		thisCard->csd.READ_BL_LEN= receiveBuffer[6] & 0x0F;
		thisCard->csd.C_SIZE=(((unsigned long)receiveBuffer[7] & 0x03)<<10) + ((unsigned long)receiveBuffer[8]<<2) + ((unsigned long)receiveBuffer[9]>>6);
		thisCard->csd.C_SIZE_MULT=(((unsigned short)receiveBuffer[10] & 0x03)<<1) + ((unsigned short)receiveBuffer[11]>>7);
		thisCard->csd.SECTOR_SIZE=(((unsigned short)receiveBuffer[11] & 0x3F)<<1) + ((unsigned short)receiveBuffer[12]>>7);
		thisCard->csd.FILE_FORMAT=(receiveBuffer[15]);

		// now we compute the size in KB = 1024 bytes
		// for this version of the CSD register (ver 1.0) we compute the size in BYTES as follows:
		// size (BYTES) = (C_SIZE+1)* 2^(C_SIZE_MULT+2) * 2^(READ_BL_LEN)

		// since READ_BL_LEN>=9 and we add 2 then minus 10 for the KB!
		//thisCard->SIZE=(unsigned long)(1<<((thisCard->csd.READ_BL_LEN)+(thisCard->csd.C_SIZE_MULT)-8))*(thisCard->csd.C_SIZE+1);
		thisCard->SIZE=(unsigned long)(((unsigned long)thisCard->csd.C_SIZE+1)*(unsigned long)((unsigned long)1<<(thisCard->csd.C_SIZE_MULT+thisCard->csd.READ_BL_LEN-8)));
	} 
	else
	if((receiveBuffer[1] & 0xC0)==0x40)
	{
		// for Ver 2.0 SDHC Card shows a 01b in the first two bits of the CSD register
		thisCard->csd.TAAC=receiveBuffer[2];	
		thisCard->csd.NSAC=receiveBuffer[3];
		thisCard->csd.TRAN_SPEED=receiveBuffer[4];
		thisCard->csd.CCC=((unsigned short)receiveBuffer[5]<<4)+((unsigned short)receiveBuffer[6]>>4);
		thisCard->csd.READ_BL_LEN= (unsigned short)receiveBuffer[6] & 0x0F;	
		thisCard->csd.C_SIZE= ((unsigned long)((unsigned long)receiveBuffer[8] & 0x3F)<<16)+((unsigned long)receiveBuffer[9]<<8)+(unsigned long)receiveBuffer[10];
		thisCard->csd.C_SIZE_MULT=0x07;			// hard coded
		thisCard->csd.SECTOR_SIZE=((unsigned short)(receiveBuffer[11] & 0x3F)<<1) + ((unsigned short)receiveBuffer[12]>>7);
		thisCard->csd.FILE_FORMAT=(receiveBuffer[15]);

		// now we compute the size in KB = 1024 bytes
		// for this version of the CSD register (ver 2.0) we compute the size in BYTES as follows:
		// size (BYTES) = (C_SIZE+1) * 512 * 1024
		thisCard->SIZE=(unsigned long)(thisCard->csd.C_SIZE + 1)*512;
	}
	else
	if((receiveBuffer[1] & 0xC0)==0x80)
	{	
		// MMC cards
		thisCard->csd.TAAC=receiveBuffer[2];	
		thisCard->csd.NSAC=receiveBuffer[3];
		thisCard->csd.TRAN_SPEED=receiveBuffer[4];
		thisCard->csd.CCC=((unsigned short)receiveBuffer[5]<<4)+((unsigned short)receiveBuffer[6]>>4);
		thisCard->csd.READ_BL_LEN= receiveBuffer[6] & 0x0F;	
		thisCard->csd.C_SIZE=(((unsigned long)receiveBuffer[7] & 0x03)<<10) + ((unsigned long)receiveBuffer[8]<<2) + ((unsigned long)receiveBuffer[9]>>6);
		thisCard->csd.C_SIZE_MULT=((unsigned short)(receiveBuffer[10] & 0x03)<<1) + ((unsigned short)receiveBuffer[11]>>7);
		i=(receiveBuffer[11]>>2) & 0x1F;							// i represents the 5 bit structure ERASE_GRP_SIZE
		j=((receiveBuffer[11] & 0x03)<<3)+(receiveBuffer[12]>>5);	// j represents the 5 bit structure ERASE_GRP_MULT
		thisCard->csd.SECTOR_SIZE=(i+1)*(j+1);						// how it works out for MMC
		thisCard->csd.FILE_FORMAT=(receiveBuffer[15]);
		// now we compute the size in KB = 1024 bytes
		// for this version of the CSD register (MMC card) we compute the size in BYTES as follows:
		// size (BYTES) = (C_SIZE+1) * 2^(C_SIZE_MULT+2) * 2^(READ_BL_LEN)
		thisCard->SIZE=(unsigned long)(((unsigned long)1<<(thisCard->csd.READ_BL_LEN+thisCard->csd.C_SIZE_MULT-8))*((unsigned long)thisCard->csd.C_SIZE+1)); 
	} else
	{
		// unknown CSD 2 bit header = 11b (?!)
		thisCard->csd.TAAC=0;	
		thisCard->csd.NSAC=0;
		thisCard->csd.TRAN_SPEED=0;
		thisCard->csd.CCC=0;
		thisCard->csd.READ_BL_LEN=0;	
		thisCard->csd.C_SIZE=0;
		thisCard->csd.C_SIZE_MULT=0;
		thisCard->csd.SECTOR_SIZE=0;
		thisCard->csd.FILE_FORMAT=0;
		thisCard->SIZE=0;
	}
	return 1;
}

int getCIDCSDRegisters(CARD_INFO* thisCard)
{
	getCIDRegister(thisCard);
	getCSDRegister(thisCard);
	return 1;
}

#endif

#if(DEBUG_SD_INIT)
void showReceiveBuffer(int x)
{
	int i;
	unsigned char receiveBuffer[20];	

	#if (DEBUG_SD_INIT)
	for(i=0; i<x; i++)
	{
		disAUART(receiveBuffer[i]);
		putcUART('.');
	}
	putrsUART((const   rom char*)"\r\n");
	
	#endif
	
}
#endif

int InitSDMMCCardSPI(CARD_INFO* thisCard)
{
	TIMEOUT_TYPE  i;
	unsigned char response;
	unsigned char receiveBuffer[20];

	#if(DEBUG_SD_INIT)
		putrsUART((const   rom char*)"Initialising SD Card v.2\r\n");
	#endif
	
	thisCard->cERROR=ERROR_NOT_SDMMC_CARD;
	InitSPI(0x00);
	SDCS=0;
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);	
	SDCS=1;
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	WriteSPIWithoutSS(0xFF);
	thisCard->checkCRC=0;
	receiveBuffer[0]=0;
	receiveBuffer[1]=0;
	receiveBuffer[2]=0;
	receiveBuffer[3]=0;
	sendCommandSDCardSPI(CMD0, CMD0_R, &receiveBuffer[0], &receiveBuffer[0]);
	receiveBuffer[0]=0x00;
	receiveBuffer[1]=0x00;
	receiveBuffer[2]=0x01;	// [11:8]=VHS   0001b = 2.7 - 3.6V voltage range from Host
	receiveBuffer[3]=0xAA;	// check Pattern
	sendCommandSDCardSPI(CMD8, CMD8_R, &receiveBuffer[0], &receiveBuffer[0]);
	if(receiveBuffer[0] & 4)
	{
		thisCard->CARD_TYPE=2;			// assume SD v1 first
		// Now Follow Version 1 branch of initialisation procedure
		#if(DEBUG_SD_INIT)
			putrsUART((const   rom char*)"v1\r\n");
		#endif	
		receiveBuffer[0]=0;
		receiveBuffer[1]=0;
		receiveBuffer[2]=0;
		receiveBuffer[3]=0;
		// read OCR command CMD58 (while not mandatory, it is recommended to do this)
		sendCommandSDCardSPI(CMD58, CMD58_R, &receiveBuffer[0], &receiveBuffer[0]);
		thisCard->ocr.OCR[0]=receiveBuffer[1];
		thisCard->ocr.OCR[1]=receiveBuffer[2];
		thisCard->ocr.OCR[2]=receiveBuffer[3];
		thisCard->ocr.OCR[3]=receiveBuffer[4];
				
		#if(DEBUG_SD_INIT)
			showReceiveBuffer(5);
		#endif
			
		if(receiveBuffer[0] & 4)
		{	
			// illegal command response, so it is not an SD or MMC Card!
			thisCard->CARD_TYPE=0;
			thisCard->cERROR=ERROR_NOT_SDMMC_CARD;
			// abort!
			return 0;
		}
		
		#if(DEBUG_SD_INIT)
			putrsUART((const   rom char*)"Checking 3.3V Operation (1)\r\n");
		#endif
	
		if(thisCard->ocr.OCR[1] & 0x10)thisCard->VOLTAGE_RANGE=1;
		else
		{
			// this card does not support the 3.3V voltage range
			thisCard->VOLTAGE_RANGE=0;
			thisCard->cERROR=ERROR_BAD_VOLTAGE_RANGE;
			// abort!
			return 0;
		}
	
		i=0;
		do
		{
		receiveBuffer[0]=0;
		receiveBuffer[1]=0;
		receiveBuffer[2]=0;
		receiveBuffer[3]=0;
		sendCommandSDCardSPI(CMD55,  CMD55_R,  &receiveBuffer[0], &receiveBuffer[0]);
		receiveBuffer[0]=0;
		receiveBuffer[1]=0;
		receiveBuffer[2]=0;
		receiveBuffer[3]=0;
		sendCommandSDCardSPI(ACMD41, ACMD41_R, &receiveBuffer[0], &receiveBuffer[0]);
		if(receiveBuffer[0] & 4)thisCard->CARD_TYPE=0;
		//DelayMs(5);
		i++;
		} while ((receiveBuffer[0] & 1) && (i<SDMMC_IDLE_WAIT_MAX));

		if(i>=SDMMC_IDLE_WAIT_MAX)
		{
			// MMC cards should end up here!
			thisCard->CARD_TYPE=1;			// MMC
					
			#if(DEBUG_SD_INIT)
				putrsUART((const   rom char*)"Not an SD Card. Sending CMD1.\r\n");
			#endif
		
			i=0;
			do
			{
			receiveBuffer[0]=0x00;
			receiveBuffer[1]=0x00;
			receiveBuffer[2]=0x00;
			receiveBuffer[3]=0x00;
			sendCommandSDCardSPI(CMD1, CMD1_R, &receiveBuffer[0], &receiveBuffer[0]);
		
			#if(DEBUG_SD_INIT)
				showReceiveBuffer(1);
			#endif
		
			//DelayMs(5);
			i++;
			} while((receiveBuffer[0] & 1) && (i<SDMMC_IDLE_WAIT_MAX));
			
			if(i>=SDMMC_IDLE_WAIT_MAX)
			{
				// initialization timed out!
				thisCard->VOLTAGE_RANGE=0;	
				thisCard->cERROR=ERROR_NOT_SDMMC_CARD;
				return 0;
			}
		
			#if(DEBUG_SD_INIT)
				putrsUART((const   rom char*)"Reading OCR.\r\n");
			#endif
			
			receiveBuffer[0]=0;
			receiveBuffer[1]=0;
			receiveBuffer[2]=0;
			receiveBuffer[3]=0;
			// read OCR command CMD58 (while not mandatory, it is recommended to do this)
			sendCommandSDCardSPI(CMD58, CMD58_R, &receiveBuffer[0], &receiveBuffer[0]);
			thisCard->ocr.OCR[0]=receiveBuffer[1];
			thisCard->ocr.OCR[1]=receiveBuffer[2];
			thisCard->ocr.OCR[2]=receiveBuffer[3];
			thisCard->ocr.OCR[3]=receiveBuffer[4];
			
			#if(DEBUG_SD_INIT)
				showReceiveBuffer(5);
			#endif
		
			if(thisCard->ocr.OCR[1] & 0x10)thisCard->VOLTAGE_RANGE=1;
			else
			{
			// this card does not support the 3.3V voltage range
			thisCard->VOLTAGE_RANGE=0;
			thisCard->cERROR=ERROR_BAD_VOLTAGE_RANGE;
			// abort!
			return 0;
			}
#if(USE_SD_REGISTERS!=0)
			getCIDCSDRegisters(thisCard);
#endif
			// Turn off CRC16 checking to speed up reads and writes...
			receiveBuffer[0]=0;
			receiveBuffer[1]=0;
			receiveBuffer[2]=0;
			receiveBuffer[3]=0;
			sendCommandSDCardSPI(CMD59, CMD59_R, &receiveBuffer[0], &receiveBuffer[0]);
			if(receiveBuffer[0]==0)thisCard->checkCRC=0;
			thisCard->cERROR=ERROR_ALL_OK;
			InitSPI(SPI_SPEED);
			// abort!
			return 1;			
		}
#if(USE_SD_REGISTERS!=0)
		getCIDCSDRegisters(thisCard);
#endif
		// Turn off CRC16 checking to speed up reads and writes...
		receiveBuffer[0]=0;
		receiveBuffer[1]=0;
		receiveBuffer[2]=0;
		receiveBuffer[3]=0;
		sendCommandSDCardSPI(CMD59, CMD59_R, &receiveBuffer[0], &receiveBuffer[0]);
		if(receiveBuffer[0]==0)thisCard->checkCRC=0;
		thisCard->cERROR=ERROR_ALL_OK;
		InitSPI(SPI_SPEED);
		return 1;
	} 
	else
	{
		#if(DEBUG_SD_INIT)
			putrsUART((const   rom char*)"v2\r\n");
		#endif
	
		thisCard->CARD_TYPE=4;				// SD v 2
		thisCard->VOLTAGE_RANGE=1;
		if(receiveBuffer[3]!=0x01)
		{
			// no echo back from CMD8 so voltage range is not compatible!
			thisCard->VOLTAGE_RANGE=0;
			thisCard->cERROR=ERROR_BAD_VOLTAGE_RANGE;
			// abort!
			return 0;
		}
		
		#if(DEBUG_SD_INIT)
			putrsUART((const   rom char*)"Reading OCR.\r\n");
		#endif
			
		receiveBuffer[0]=0;
		receiveBuffer[1]=0;
		receiveBuffer[2]=0;
		receiveBuffer[3]=0;
		// read OCR command CMD58 (while not mandatory, it is recommended to do this)
		sendCommandSDCardSPI(CMD58, CMD58_R, &receiveBuffer[0], &receiveBuffer[0]);
		thisCard->ocr.OCR[0]=receiveBuffer[1];
		thisCard->ocr.OCR[1]=receiveBuffer[2];
		thisCard->ocr.OCR[2]=receiveBuffer[3];
		thisCard->ocr.OCR[3]=receiveBuffer[4];

		#if(DEBUG_SD_INIT)
			showReceiveBuffer(5);
		#endif
		
		if(receiveBuffer[0] & 4)
		{	
			// illegal command response, so it is not an SD Card!
			thisCard->cERROR=ERROR_NOT_SDMMC_CARD;
			// abort!
			return 0;
		}
		
		#if(DEBUG_SD_INIT)
			putrsUART((const   rom char*)"Checking 3.3V Operation. (2)\r\n");
		#endif
		
		if(thisCard->ocr.OCR[1] & 0x10)thisCard->VOLTAGE_RANGE=1;
		else
		{
			// this card does not support the 3.3V voltage range
			thisCard->VOLTAGE_RANGE=0;
			thisCard->cERROR=ERROR_BAD_VOLTAGE_RANGE;
			// abort!
			return 0;
		}
		i=0;
		do
		{
		receiveBuffer[0]=0x0;
		receiveBuffer[1]=0x0;
		receiveBuffer[2]=0x0;
		receiveBuffer[3]=0x0;
		sendCommandSDCardSPI(CMD55,  CMD55_R,  &receiveBuffer[0], &receiveBuffer[0]);
		receiveBuffer[0]=0x40;			// HCS (bit 30)=1 if host supports High Capacity >2Gb - 32Gb
		receiveBuffer[1]=0x0;
		receiveBuffer[2]=0x0;
		receiveBuffer[3]=0x0;
		sendCommandSDCardSPI(ACMD41, ACMD41_R, &receiveBuffer[0], &receiveBuffer[0]);
		response=receiveBuffer[0];
		
		#if(DEBUG_SD_INIT)
			putrsUART((const   rom char*)"Reading OCR (CCS).\r\n");
		#endif
		
		receiveBuffer[0]=0;
		receiveBuffer[1]=0;
		receiveBuffer[2]=0;
		receiveBuffer[3]=0;
		sendCommandSDCardSPI(CMD58, CMD58_R, &receiveBuffer[0], &receiveBuffer[0]);
		thisCard->ocr.OCR[0]=receiveBuffer[1];
		thisCard->ocr.OCR[1]=receiveBuffer[2];
		thisCard->ocr.OCR[2]=receiveBuffer[3];
		thisCard->ocr.OCR[3]=receiveBuffer[4];
		
		#if(DEBUG_SD_INIT)
			showReceiveBuffer(5);
		#endif
		
		if(receiveBuffer[0] & 4)thisCard->CARD_TYPE=0;
		DelayMs(5);
		i++;
		} while ((response & 1) && (i<SDMMC_IDLE_WAIT_MAX));

		if(i>=SDMMC_IDLE_WAIT_MAX)
		{
			thisCard->cERROR=ERROR_SDMMC_CARD_TIMEOUT;
			// abort!
			return 0;			
		}
		if(thisCard->ocr.OCR[0] & 0x40)thisCard->CARD_TYPE=6;			// this is a High Capacity SD Card! ie. CCS (bit 30 of OCR) = 1

#if(USE_SD_REGISTERS!=0)
		getCIDCSDRegisters(thisCard);
#endif

		// Turn off CRC16 checking to speed up reads and writes...
		receiveBuffer[0]=0;
		receiveBuffer[1]=0;
		receiveBuffer[2]=0;
		receiveBuffer[3]=0;
		sendCommandSDCardSPI(CMD59, CMD59_R, &receiveBuffer[0], &receiveBuffer[0]);

		if(receiveBuffer[0]==0)
		{
			thisCard->checkCRC=0;
		}

		thisCard->cERROR=ERROR_ALL_OK;
		InitSPI(SPI_SPEED);
		return 1;
	}
	return 0;
}

unsigned char* getMemoryCardPNM(CARD_INFO* cInfo, unsigned char* outstr)
{
	*outstr++=cInfo->cid.PNM[0];
	*outstr++=cInfo->cid.PNM[1];
	*outstr++=cInfo->cid.PNM[2];
	*outstr++=cInfo->cid.PNM[3];
	*outstr++=cInfo->cid.PNM[4];
	*outstr++=cInfo->cid.PNM[5];
	*outstr++=cInfo->cid.PNM[6];
	*outstr='\0';
	return outstr;
}

unsigned char* getMemoryCardPSN(CARD_INFO* cInfo, unsigned char* outstr)
{
	long l;
	l=(((long)cInfo->cid.PSN[3])<<24)+(((long)cInfo->cid.PSN[2])<<16)+(((long)cInfo->cid.PSN[1])<<8)+((long)cInfo->cid.PSN[0]);
	return myultoa(l, outstr, 0);
}

unsigned char* getMemoryCardPRV(CARD_INFO* cInfo, unsigned char* outstr)
{
	long l;
	l=(long)cInfo->cid.PRV;
	return myultoa(l, outstr, 0);
}

unsigned char* getMemoryCardType(CARD_INFO* cInfo, unsigned char* outstr)
{
	switch(cInfo->CARD_TYPE)
	{
		case 1:
			outstr=copyStringRom(outstr, (const rom unsigned char*)"MMC");
			break;
			
		case 2:
			outstr=copyStringRom(outstr, (const rom unsigned char*)"SD v.1");
			break;
			
		case 4:
			outstr=copyStringRom(outstr, (const rom unsigned char*)"SD v.2");
			break;
			
		case 6:
			outstr=copyStringRom(outstr, (const rom unsigned char*)"SDHC");
			break;
			
		default:
			outstr=copyStringRom(outstr, (const rom unsigned char*)"None");
			break;
	}
	*outstr='\0';
	return outstr;
}

unsigned char* getMemoryCardFreeSize(CARD_INFO* cInfo, unsigned char* outstr)
{
	outstr=getFileSizeString((float)cInfo->SIZEFREE*1024.0, outstr);
	return outstr;
}

unsigned char* getMemoryCardTotalSize(CARD_INFO* cInfo, unsigned char* outstr)
{
	outstr=getFileSizeString((float)cInfo->SIZE*1024.0, outstr);
	return outstr;
}

FRESULT updateFreeSize(CARD_INFO* thisCard)
{
	FRESULT fresult;
	FATFS*  fpointer;
	DWORD   d;
	char    str[2];
	
	str[0]='/';
	str[1]='\0';
	fpointer=&fileSystem;
	thisCard->SIZEFREE=0;
	if(memoryCardSystemUpMacro())
	{
		fresult=f_getfree(str, (DWORD*)&d, &fpointer);
		if(fresult==FR_OK)
		{
			thisCard->SIZEFREE=(unsigned long)((fpointer->max_clust - 2) * fpointer->csize/2);
		} 
	}
	return fresult; 
}

FRESULT initMemoryCardSystem(CARD_INFO* thisCard)
{
	FRESULT 		result;
	unsigned int 	written;
	
	memoryCardSystemUp=0;
	/* Initialize SD card and FATFS */
	InitSDMMCCardSPI(thisCard);
	f_InitSystem();
	result=f_mount(0, &fileSystem);
	if((result==FR_OK)&&(thisCard->cERROR==ERROR_ALL_OK))
	{
		memoryCardSystemUp=1;
		/* check settings and saved EE settings? */
	}
	return result;
}
